/*
	This is a part of the source code for Pro/DESKTOP.
	Copyright (C) 2002 Parametric Technology Corporation.
	All rights reserved.
*/

#include "stdafx.h"

#include "TableDimAddin.h"
#include "ProdAddInCls.h"

#define CHECK_RETURN_STATUS(status, ret) \
if (FAILED(status) || ret == NULL) { \
	return status; \
} \

#define GET_CLASS_OBJECT(classObj, className) \
	pDisp = NULL;\
	status = pApp->GetClass(className, &pDisp); \
	CHECK_RETURN_STATUS(status, pDisp);\
	\
	classObj = pDisp; \
	pDisp->Release(); \
	\
	if (classObj == NULL) \
		return S_FALSE;\

IApplicationPtr pApp(__uuidof(ProDESKTOP));

//------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP ProdAddInCls::OnStartUp()
{
	IProDExtensibilityPtr prodAdd(pApp);

	IDispatch *pDispatch;
	QueryInterface(IID_IDispatch, (void**) &pDispatch);
	pDispatch->AddRef();

	CComVariant context, handler;
	handler.vt = context.vt = VT_DISPATCH;
	context.pdispVal = handler.pdispVal = pDispatch;

	CComBSTR resString;
	IUserCommand *dimsToTable;

	resString.LoadString(IDS_Dimensions);
	prodAdd->AddUserCommand(DrawingMenuBar, DrawingTableMenu, -1, resString, context, handler, &dimsToTable);	// adds the command in menu

	dimsToTable->SetIdentifier(2000);

	resString.LoadString(IDS_SelectedDimension);
	dimsToTable->SetPrompt(resString);

	resString.LoadString(IDS_Accelerator);
	dimsToTable->SetAccelerator(resString);

	return S_OK;
}

STDMETHODIMP ProdAddInCls::OnCloseDown()
{
	IProDExtensibilityPtr prodAdd(pApp);

	CComBSTR resString;
	resString.LoadString(IDS_Dimensions);

	IUserCommand *dimsToTable;
	prodAdd->GetUserCommand(DrawingMenuBar, DrawingTableMenu, resString, &dimsToTable);
	prodAdd->RemoveUserCommand(dimsToTable);

	return S_OK;
}

//------------------------------------------------------------------------------------------------------------------------
static HRESULT CreateDimTable(ISheetPtr pSheet, int nColumns, ITablePtr &pTable)
{
	IDispatch *pDisp = NULL;
	HRESULT status;

	IVectorClassPtr pVectorClass;
	GET_CLASS_OBJECT(pVectorClass, CComBSTR("Vector"))

	IVectorPtr pVector;
	status = pVectorClass->CreateVector(0.0, 0.0, 0.0, &pVector);	// we'll set the actual position later
	CHECK_RETURN_STATUS(status, pVector)

	double textHeight = 0.005;	// 5mm
	double columnWidth = 0.05;	// 50mm

	ITableClassPtr pTableClass;
	GET_CLASS_OBJECT(pTableClass, CComBSTR("Table"))

	status = pTableClass->CreateTable(pVector, textHeight, nColumns, columnWidth, alignCenter, &pTable);
	CHECK_RETURN_STATUS(status, pTable)

	pSheet->AddTable(pTable);
	pTable->SetInverted(VARIANT_FALSE);

	double width, height;
	pSheet->GetWidth(&width);
	pSheet->GetHeight(&height);

	status = pVectorClass->CreateVector(width, height, 0.0, &pVector);
	CHECK_RETURN_STATUS(status, pVector)

	pTable->SetAnchor(anchorTopLeft);
	pTable->SetAnchorPoint(anchorTopLeft, pVector);

	long nodeIndex = 0, indent = 1;

	int retVal;
	pTable->SetIndent(nodeIndex, indent, &retVal);
	pTable->SetTitle(nodeIndex, CComBSTR("Tabular Dimensions"));

	nodeIndex = 1;
	pTable->SetIndent(nodeIndex, indent, &retVal);
	pTable->SetTitle(nodeIndex, CComBSTR("Name"));

	nodeIndex = 2;
	pTable->SetIndent(nodeIndex, indent, &retVal);
	pTable->SetTitle(nodeIndex, CComBSTR("Value"));

	return S_OK;
}

static bool GetLabelLetter(int i, char *label)
{
	if (i > 48) {
		::MessageBox(NULL, "Number of dimensions exceeded", "TableDimAddin", MB_OK);
		return false;
	}

	i += 65;

	if (i > 90 - 3) {	// adjust if past Z(90), taking into account adjustments below
		i -= 90 - 3 - 64;
		sprintf(label, "%c%c\n", i, 'A');
		return true;
	}

	if (i >= 73 && i < 90 ) {
		i++;

		if (i >= 79 && i < 90 ) {
			i++;

			if (i >= 81 && i < 90)
				i++;
		}
	}

	sprintf(label, "%c\n", i);
	return true;
}

static HRESULT InsertRow(IDrawingPtr pDrawing, ITablePtr pTable, IDimensionCalloutPtr pDimCallout)
{
	pTable->InsertRow(-1);

	int retVal;
	pTable->GetRowCount(&retVal);
	long row = retVal - 1;

	int nodes;
	pTable->GetNodeCount(&nodes);
	
	ITableColumnPtr pLabelColumn, pValueColumn;

	HRESULT status = pTable->GetColumn(1, &pLabelColumn);
	CHECK_RETURN_STATUS(status, pLabelColumn)

	status = pTable->GetColumn(2, &pValueColumn);
	CHECK_RETURN_STATUS(status, pValueColumn)

	ITableCellPtr pLabelCell, pValueCell;

	status = pLabelColumn->GetCell(row, &pLabelCell);
	CHECK_RETURN_STATUS(status, pLabelCell)

	status = pValueColumn->GetCell(row, &pValueCell);
	CHECK_RETURN_STATUS(status, pValueCell)

	char string[10];
	if (!GetLabelLetter(row, string))
		return S_FALSE;

	CComBSTR label(strlen(string), string);
	IDispatch *pDisp = NULL;

	// create a note
	INoteClassPtr pNoteClass;
	GET_CLASS_OBJECT(pNoteClass, CComBSTR("Note"))

	INotePtr pNote;
	status = pNoteClass->CreateNote(pDrawing, label, &pNote);
	CHECK_RETURN_STATUS(status, pNote)

	// create a note callout
	INoteCalloutClassPtr pNoteCalloutClass;
	GET_CLASS_OBJECT(pNoteCalloutClass, CComBSTR("NoteCallout"))

	INoteCalloutPtr pNoteCallout;
	status = pNoteCalloutClass->CreateNoteCallout(pNote, &pNoteCallout);
	CHECK_RETURN_STATUS(status, pNoteCallout)

	// create vector
	IVectorClassPtr pVectorClass;
	GET_CLASS_OBJECT(pVectorClass, CComBSTR("Vector"))

	IVectorPtr pVector;
	status = pVectorClass->CreateVector(0.0, 0.0, 0.0, &pVector);	//we'll set the actual position later
	CHECK_RETURN_STATUS(status, pVector)

	// create lable callout group
	ICalloutGroupClassPtr pCalloutGroupClass;
	GET_CLASS_OBJECT(pCalloutGroupClass, CComBSTR("CalloutGroup"))

	ICalloutGroupPtr pLabelCalloutGroup, pValueCalloutGroup;
	double height;
	pTable->GetTextHeight(&height);
	ICalloutPtr pLabelCallout(pNoteCallout);

	status = pCalloutGroupClass->CreateCalloutGroup(pLabelCallout, pVector, height, &pLabelCalloutGroup);
	CHECK_RETURN_STATUS(status, pLabelCalloutGroup)

	// sets the label callout group in label cell
	pLabelCell->SetCalloutGroup(pLabelCalloutGroup);

	ICalloutGroupPtr pDimCalloutGroup;
	IObjectPtr pObject(pDimCallout), pParent;
	status = pObject->GetParent(CComBSTR("CalloutGroup"), &pParent);
	pDimCalloutGroup = pParent;

	INoteCalloutPtr pLabelCallout1;
	status = pNoteCalloutClass->CreateNoteCallout(pNote, &pLabelCallout1);
	CHECK_RETURN_STATUS(status, pLabelCallout1)

	ICalloutPtr pLabel(pLabelCallout1), pDim(pDimCallout);
	pDimCalloutGroup->InsertCallout(pLabel, pDim);

	// create value callout group for value cell
	ICalloutPtr pValueCallout;
	status = pDim->CreateCopy(&pValueCallout);
	CHECK_RETURN_STATUS(status, pValueCallout)

	pObject->Delete();

	status = pCalloutGroupClass->CreateCalloutGroup(pValueCallout, pVector, height, &pValueCalloutGroup);
	CHECK_RETURN_STATUS(status, pValueCalloutGroup)

	// sets the value callout group in value cell
	pValueCell->SetCalloutGroup(pValueCalloutGroup);

	return S_OK;
}

static HRESULT PopulateTable(IDrawingPtr pDrawing, ITablePtr pTable, ISetPtr pSelection)
{
	HRESULT status;
	IDispatch *pDisp = NULL;

	IItClassPtr pItClass;
	GET_CLASS_OBJECT(pItClass, CComBSTR("it"))

	IItPtr pIt;
	status = pItClass->CreateAObjectIt(pSelection, &pIt);
	CHECK_RETURN_STATUS(status, pIt)

	pIt->start(&pDisp);
	while (pDisp) {
		IDimensionCalloutPtr pDimCallout;
		pDimCallout = pDisp;
		pDisp->Release();

		InsertRow(pDrawing, pTable, pDimCallout);
		pIt->Next(&pDisp);
	}

	return S_OK;
}

//------------------------------------------------------------------------------------------------------------------------
STDMETHODIMP ProdAddInCls::OnCommand(IUserCommand * userCommand)
{
	int identifier;
	userCommand->GetIdentifier(&identifier);
	if (identifier != 2000)
		return S_FALSE;

	IHelmPtr pHelm;
	HRESULT status = pApp->TakeHelm(&pHelm);
	CHECK_RETURN_STATUS(status, pHelm)

	IGraphicDocumentPtr pGraphicDoc;
	status = pApp->GetActiveDoc(&pGraphicDoc);
	CHECK_RETURN_STATUS(status, pGraphicDoc)

	ISetPtr pSelection;
	status = pGraphicDoc->GetSelection(CComBSTR("DimensionCallout"), &pSelection);
	CHECK_RETURN_STATUS(status, pSelection)

	long nCount = 0;
	pSelection->GetCount(&nCount);
	if (nCount == 0) {
		::MessageBox(NULL, "No dimension callouts selected", "TableDimAddin", MB_OK);
		pApp->ReleaseHelm(&pHelm);
		return S_FALSE;
	}

	IDrawingDocumentPtr pDrawingDoc(pGraphicDoc);
	if (pDrawingDoc == NULL) {
		::MessageBox(NULL, "No drawing active", "TableDimAddin", MB_OK);
		pApp->ReleaseHelm(&pHelm);
		return S_FALSE;
	}

	IDrawingPtr pDrawing;
	status = pDrawingDoc->GetDrawing(&pDrawing);
	CHECK_RETURN_STATUS(status, pDrawing)

	ISheetPtr pSheet;
	status = pDrawingDoc->GetActiveSheet(&pSheet);
	CHECK_RETURN_STATUS(status, pSheet)

	ITablePtr pTable;
	CreateDimTable(pSheet, 2, pTable);	// only 2 columns
	PopulateTable(pDrawing, pTable, pSelection);

	pHelm->CommitCalls(CComBSTR("Create Table"), VARIANT_FALSE);
	pApp->ReleaseHelm(&pHelm);
	pGraphicDoc->ClearSelection();

	return S_OK;
}

STDMETHODIMP ProdAddInCls::OnUpdateCommand(IUserCommand * userCommand)
{
	int identifier;
	userCommand->GetIdentifier(&identifier);
	if (identifier != 2000)
		return S_FALSE;

	IGraphicDocumentPtr pGraphicDoc;
	HRESULT status = pApp->GetActiveDoc(&pGraphicDoc);
	CHECK_RETURN_STATUS(status, pGraphicDoc)

	ISetPtr pSelection;
	status = pGraphicDoc->GetSelection(CComBSTR("DimensionCallout"), &pSelection);
	CHECK_RETURN_STATUS(status, pSelection)

	long nCount = 0;
	pSelection->GetCount(&nCount);
	userCommand->SetMenuState(nCount ? 1 : 0);

	return S_OK;
}